/* $Id: timer.c,v 1.10 1998/09/16 18:13:05 ericb Exp $ */
/* Copyright (C) 1997 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

#include "spil_d.h"
#include "e1432.h"

/* Normally this code uses the standard xfer_lbus_to_ram to transfer
   data from local bus to main RAM.  If FIT_TO_RAM is defined, it
   instead uses lower-level function calls to do the same thing.
   FIT_TO_RAM is slightly more efficient, improving the transfer rate
   by ~5% when the blocksize is 8192.  The savings is relatively small
   and gets smaller as the blocksize gets bigger. */
/*#define	FIT_TO_RAM*/

#define	NMOD			1
#define	NCHAN			8
#define	NSCAN			10
#define	BLOCKSIZE		8192
#define	CLOCK_FREQ		65536.0
#define	SPAN			25600.0

#define	NMOD_MAX		8
#define	NCHAN_MAX		128

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) printf("%s: returned %d\n", #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) printf("%s: returned %d\n", #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: timer.c,v 1.10 1998/09/16 18:13:05 ericb Exp $";

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].input_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) printf("e1432_create_channel_group: returned %d\n",
		      *group);
	return -1;
    }

    return 0;
}

static int
setup(E1432ID hw, int group, int nchan)
{
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));
    CHECK(e1432_set_clock_freq(hw, group, CLOCK_FREQ));
    CHECK(e1432_set_data_mode(hw, group, E1432_BLOCK_MODE));
    CHECK(e1432_set_data_port(hw, group, E1432_SEND_PORT_LBUS));
    CHECK(e1432_set_lbus_mode(hw, group, E1432_LBUS_MODE_APPEND));
    CHECK(e1432_set_lbus_mode(hw, E1432_INPUT_CHAN(1),
			      E1432_LBUS_MODE_GENERATE));
    CHECK(e1432_set_span(hw, group, SPAN));
    return 0;
}

static int
start(E1432ID hw, int group, int nchan)
{
    int     chan;

    CHECK(e1432_reset_lbus(hw, group, E1432_RESET_LBUS_ON));
    CHECK(lbus_control(LBUS_CTL_RESET, 1));

    /* Unreset all channels in order, assuming that the channel order
       matches the module order in the mainframe.  This is just an
       assumption, since we can't determine the true slot numbers
       corresponding to each module.  This calls e1432_reset_lbus for
       each channel when it really only needs to be called for each
       module.  This is slightly inefficient, but it's not that
       bad. */
    for (chan = 0; chan < nchan; chan++)
	CHECK(e1432_reset_lbus(hw, E1432_INPUT_CHAN(chan + 1),
			       E1432_RESET_LBUS_OFF));

    CHECK(lbus_control(LBUS_CTL_RESET, 0));
#ifdef	FIT_TO_RAM
    CHECK(lbus_setup_consume(LBUS_CONSUME_CONT, LBUS_WIDTH_32, 0));
#endif
    CHECK(e1432_init_measure(hw, group));
    return 0;
}

static int
run(E1432ID hw, int group, int nchan)
{
    unsigned long start, end, sum;
    short   buf[BLOCKSIZE];
    int     error, chan, scan;

    if (ptimer_init() < 0)
    {
	CHECK(ptimer_stop());
	CHECK(ptimer_init());
    }
    CHECK(ptimer_set_us(USEC100));

    sum = 0;
    for (scan = 0; scan < NSCAN; scan++)
    {
	while ((error = e1432_block_available(hw, group)) == 0)
	    opsys_suspend();
	if (error < 0)
	    (void) printf("e1432_block_available: returned %d\n", error);

	start = ptimer_read();
#ifdef	FIT_TO_RAM
	CHECK(fit_setup_xfer(FIT_SETUP_N_START, FIT_DEV_LBUS,
			     FIT_MAP_DEST(FIT_DEV_CPU),
			     FIT_SETUP_NO_COUNT
			     /*nchan * BLOCKSIZE / 2 can get too big*/));
#endif
	for (chan = 0; chan < nchan; chan++)
	{
#ifdef	FIT_TO_RAM
	    CHECK(xfer_fit_to_ram(FIT_TO_MAIN32, buf, BLOCKSIZE / 2));
#else
	    CHECK(xfer_lbus_to_ram(LBUS_TO_MAIN32, buf, 1, BLOCKSIZE / 2));
#endif
	}

	end = ptimer_read();
	sum += end - start;
    }

    CHECK(ptimer_stop());

    (void) printf("timer %s\n", rcsid);

    (void) printf("Channels:  %6d\n", nchan);
    (void) printf("Blocksize: %6d\n", BLOCKSIZE);
    (void) printf("Span:      %6g Hz\n", SPAN);
    (void) printf("Xfer Rate: %6.2f MB/sec\n",
		  nchan * NSCAN * BLOCKSIZE * 2 /
		  (sum * 100.0e-6 * 1000000.0));

    (void) printf("Sending results to host\n");
    (void) host_send_msg(MSG_NO_WAIT, nchan);
    (void) host_send_msg(MSG_WAIT, BLOCKSIZE);
    (void) host_send_msg(MSG_WAIT, SPAN);
    (void) host_send_msg(MSG_WAIT, nchan * NSCAN * BLOCKSIZE * 2 /
			 (sum * 100.0e-6));
    (void) printf("Done\n");

    return 0;
}

int
main(void)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    int     nchan = NCHAN;
    int     i, group;

    for (i = 0; i < NMOD; i++)
	laddr[i] = 129 + i;

    /* Use exit, not return, as required by SPOS */
    if (init(NMOD, laddr, &hw, &group, &nchan, chan_list) < 0)
	exit(1);
    if (setup(hw, group, nchan) < 0)
	exit(1);
    if (start(hw, group, nchan) < 0)
	exit(1);
    if (run(hw, group, nchan) < 0)
	exit(1);
    exit(0);
}
